/*
* mhal_pinmux.c- Sigmastar
*
* Copyright (C) 2018 Sigmastar Technology Corp.
*
* Author: karl.xiao <karl.xiao@sigmastar.com.tw>
*
* This software is licensed under the terms of the GNU General Public
* License version 2, as published by the Free Software Foundation, and
* may be copied, distributed, and modified under those terms.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
*/
#include "ms_platform.h"
#include "mdrv_types.h"
#include "mhal_gpio.h"
#include "padmux.h"
#include "gpio.h"

//==============================================================================
//
//                              MACRO DEFINE
//
//==============================================================================

#define BASE_RIU_PA                         0xFD000000
#define PMSLEEP_BANK                        0x000E00
#define SAR_BANK                            0x001400
#define ALBANY1_BANK                        0x003200
#define ALBANY2_BANK                        0x003300
#define CHIPTOP_BANK                        0x101E00
#define UTMI0_BANK                          0x142100

#define _GPIO_W_WORD(addr,val)              {(*(volatile u16*)(addr)) = (u16)(val);}
#define _GPIO_W_WORD_MASK(addr,val,mask)    {(*(volatile u16*)(addr)) = ((*(volatile u16*)(addr)) & ~(mask)) | ((u16)(val) & (mask));}
#define _GPIO_R_BYTE(addr)                  (*(volatile u8*)(addr))
#define _GPIO_R_WORD_MASK(addr,mask)        ((*(volatile u16*)(addr)) & (mask))

#define GET_BASE_ADDR_BY_BANK(x, y)         ((x) + ((y) << 1))
#define _RIUA_8BIT(bank , offset)           GET_BASE_ADDR_BY_BANK(BASE_RIU_PA, bank) + (((offset) & ~1)<<1) + ((offset) & 1)
#define _RIUA_16BIT(bank , offset)          GET_BASE_ADDR_BY_BANK(BASE_RIU_PA, bank) + ((offset)<<2)

/* Non PM Pad : CHIPTOP_BANK */
#define REG_PWM5_MODE           0x02
    #define REG_PWM5_MODE_MASK      BIT2|BIT1|BIT0
#define REG_PWM6_MODE           0x02
    #define REG_PWM6_MODE_MASK      BIT5|BIT4|BIT3
#define REG_PWM7_MODE           0x02
    #define REG_PWM7_MODE_MASK      BIT8|BIT7|BIT6
#define REG_PWM8_MODE           0x02
    #define REG_PWM8_MODE_MASK      BIT11|BIT10|BIT9
#define REG_PWM9_MODE           0x02
    #define REG_PWM9_MODE_MASK      BIT14|BIT13|BIT12
#define REG_FUART_MODE          0x03
    #define REG_FUART_MODE_MASK     BIT2|BIT1|BIT0
#define REG_UART0_MODE          0x03
    #define REG_UART0_MODE_MASK     BIT6|BIT5|BIT4
#define REG_UART1_MODE          0x03
    #define REG_UART1_MODE_MASK     BIT10|BIT9|BIT8
#define REG_PWM10_MODE          0x04
    #define REG_PWM10_MODE_MASK     BIT2|BIT1|BIT0
#define REG_SR_MODE             0x06
    #define REG_SR_MODE_MASK        BIT2|BIT1|BIT0
#define REG_SR_I2C_MODE         0x06
    #define REG_SR_I2C_MODE_MASK    BIT5|BIT4
#define REG_SR_HVSYNC_MODE      0x06
    #define REG_SR_HVSYNC_MODE_MASK BIT6
#define REG_SR_MCLK_MODE        0x06
    #define REG_SR_MCLK_MODE_MASK   BIT7
#define REG_SR_PCK_MODE         0x06
    #define REG_SR_PCK_MODE_MASK    BIT8
#define REG_SR_PDN_MODE         0x06
    #define REG_SR_PDN_MODE_MASK    BIT10|BIT9
#define REG_SR_RST_MODE         0x06
    #define REG_SR_RST_MODE_MASK    BIT12|BIT11
#define REG_PWM0_MODE           0x07
    #define REG_PWM0_MODE_MASK      BIT2|BIT1|BIT0
#define REG_PWM1_MODE           0x07
    #define REG_PWM1_MODE_MASK      BIT5|BIT4|BIT3
#define REG_PWM2_MODE           0x07
    #define REG_PWM2_MODE_MASK      BIT8|BIT7|BIT6
#define REG_PWM3_MODE           0x07
    #define REG_PWM3_MODE_MASK      BIT11|BIT10|BIT9
#define REG_PWM4_MODE           0x07
    #define REG_PWM4_MODE_MASK      BIT14|BIT13|BIT12
#define REG_NAND_MODE           0x08
    #define REG_NAND_MODE_MASK      BIT0
#define REG_SD_MODE             0x08
    #define REG_SD_MODE_MASK        BIT3|BIT2
#define REG_SDIO_MODE           0x08
    #define REG_SDIO_MODE_MASK      BIT8
#define REG_I2C0_MODE           0x09
    #define REG_I2C0_MODE_MASK      BIT2|BIT1|BIT0
#define REG_I2C1_MODE           0x09
    #define REG_I2C1_MODE_MASK      BIT5|BIT4
#define REG_SPI0_MODE           0x0c
    #define REG_SPI0_MODE_MASK      BIT2|BIT1|BIT0
#define REG_SPI1_MODE           0x0c
    #define REG_SPI1_MODE_MASK      BIT6|BIT5|BIT4
#define REG_EJ_MODE             0x0f
    #define REG_EJ_MODE_MASK        BIT1|BIT0
#define REG_ETH_MODE            0x0f
    #define REG_ETH_MODE_MASK       BIT2
#define REG_CCIR_MODE           0x0f
    #define REG_CCIR_MODE_MASK      BIT5|BIT4
#define REG_TTL_MODE            0x0f
    #define REG_TTL_MODE_MASK       BIT7|BIT6
#define REG_DMIC_MODE           0x0f
    #define REG_DMIC_MODE_MASK      BIT9|BIT8
#define REG_I2S_MODE            0x0f
    #define REG_I2S_MODE_MASK       BIT11|BIT10
#define REG_TEST_IN_MODE        0x12
    #define REG_TEST_IN_MODE_MASK   BIT1|BIT0
#define REG_TEST_OUT_MODE       0x12
    #define REG_TEST_OUT_MODE_MASK  BIT5|BIT4
#define REG_EMMC_MODE           0x13
    #define REG_EMMC_MODE_MASK      BIT0
#define REG_EMMC_RSTN_EN        0x13
    #define REG_EMMC_RSTN_EN_MASK   BIT1
#define REG_MIPI_PAD_IN         0x33
    #define REG_MIPI_PAD_IN_MASK    BIT1|BIT0
#define REG_ALLPAD_IN           0x50
    #define REG_ALLPAD_IN_MASK    BIT15

/* PM Sleep : PMSLEEP_BANK */
#define REG_PM_GPIO_PM_LOCK        0x12
    #define REG_PM_GPIO_PM_LOCK_MASK   0xFFFF
#define REG_PM_GPIO_PM4_INV     0x1c
    #define REG_PM_GPIO_PM4_INV_MASK BIT1
#define REG_PM_LINK_WKINT2GPIO4 0x1c
    #define REG_PM_LINK_WKINT2GPIO4_MASK BIT3
#define REG_PM_IR_IS_GPIO       0x1c
    #define REG_PM_IR_IS_GPIO_MASK  BIT4
#define REG_PM_PWM0_MODE        0x28
    #define REG_PM_PWM0_MODE_MASK   BIT1|BIT0
#define REG_PM_PWM1_MODE        0x28
    #define REG_PM_PWM1_MODE_MASK   BIT3|BIT2
#define REG_PM_PWM2_MODE        0x28
    #define REG_PM_PWM2_MODE_MASK   BIT7|BIT6
#define REG_PM_PWM3_MODE        0x28
    #define REG_PM_PWM3_MODE_MASK   BIT9|BIT8
#define REG_PM_PWM4_MODE        0x27
    #define REG_PM_PWM4_MODE_MASK   BIT0
#define REG_PM_PWM5_MODE        0x27
    #define REG_PM_PWM5_MODE_MASK   BIT1
#define REG_PM_PWM8_MODE        0x27
    #define REG_PM_PWM8_MODE_MASK   BIT2
#define REG_PM_PWM9_MODE        0x27
    #define REG_PM_PWM9_MODE_MASK   BIT3
#define REG_PM_PWM10_MODE        0x27
    #define REG_PM_PWM10_MODE_MASK   BIT4
#define REG_PM_UART1_MODE        0x27
    #define REG_PM_UART1_MODE_MASK   BIT8
#define REG_PM_LED_MODE         0x28
    #define REG_PM_LED_MODE_MASK    BIT5|BIT4

#define REG_PM_VID_MODE         0x28
    #define REG_PM_VID_MODE_MASK    BIT13|BIT12
#define REG_PM_SD_CDZ_MODE      0x28
    #define REG_PM_SD_CDZ_MODE_MASK BIT14
#define REG_PM_SPI_IS_GPIO      0x35
    #define REG_PM_SPI_IS_GPIO_MASK BIT7|BIT6|BIT5|BIT4|BIT3|BIT2|BIT1|BIT0
    #define REG_PM_SPI_GPIO_MASK            BIT0
    #define REG_PM_SPICSZ1_GPIO_MASK        BIT2
    #define REG_PM_SPICSZ2_GPIO_MASK        BIT3
    #define REG_PM_SPIWPN_GPIO_MASK         BIT4
    #define REG_PM_SPIHOLDN_MODE_MASK       BIT6 | BIT7
#define REG_PM_SPICSZ1_GPIO     REG_PM_SPI_IS_GPIO
#define REG_PM_SPICSZ2_GPIO     REG_PM_SPI_IS_GPIO
#define REG_PM_SPI_GPIO         REG_PM_SPI_IS_GPIO
#define REG_PM_SPIWPN_GPIO      REG_PM_SPI_IS_GPIO
#define REG_PM_SPIHOLDN_MODE    REG_PM_SPI_IS_GPIO

#define REG_PM_UART_IS_GPIO     0x35
    #define REG_PM_UART_IS_GPIO_MASK BIT11|BIT10|BIT9|BIT8

/* SAR : SAR_BANK, R/W 8-bits */
#define REG_SAR_AISEL_8BIT          0x11*2
    #define REG_SAR_CH0_AISEL       BIT0
    #define REG_SAR_CH1_AISEL       BIT1
    #define REG_SAR_CH2_AISEL       BIT2
    #define REG_SAR_CH3_AISEL       BIT3

/* EMAC : ALBANY1_BANK */
#define REG_ATOP_RX_INOFF       0x69
    #define REG_ATOP_RX_INOFF_MASK  BIT15|BIT14

/* EMAC : ALBANY2_BANK */
#define REG_ETH_GPIO_EN         0x71
    #define REG_ETH_GPIO_EN_MASK    BIT3|BIT2|BIT1|BIT0

/* UTMI0 : UTMI0_BANK */
#define REG_UTMI0_FL_XVR_PDN    0x0
    #define REG_UTMI0_FL_XVR_PDN_MASK BIT12
#define REG_UTMI0_REG_PDN       0x0
    #define REG_UTMI0_REG_PDN_MASK BIT15
#define REG_UTMI0_CLK_EXTRA0_EN 0x4
    #define REG_UTMI0_CLK_EXTRA0_EN_MASK BIT7
#define REG_UTMI0_GPIO_EN       0x1f
    #define REG_UTMI0_GPIO_EN_MASK  BIT14

//-------------------- configuration -----------------
#define ENABLE_CHECK_ALL_PAD_CONFLICT       0

//==============================================================================
//
//                              STRUCTURE
//
//==============================================================================

typedef struct stPadMux
{
    U16 padID;
    U32 base;
    U16 offset;
    U16 mask;
    U16 val;
    U16 mode;
} ST_PadMuxInfo;

typedef struct stPadMode
{
    U8  u8PadName[16];
    U32 u32ModeRIU;
    U32 u32ModeMask;
} ST_PadModeInfo;

//==============================================================================
//
//                              VARIABLES
//
//==============================================================================

const ST_PadMuxInfo m_stPadMuxTbl[] =
{
    {PAD_GPIO0,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_GPIO0,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_GPIO0,     CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT6,           PINMUX_FOR_SPI1_MODE_4},
    {PAD_GPIO0,     CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT1,           PINMUX_FOR_FUART_MODE_2},
    {PAD_GPIO0,     CHIPTOP_BANK, REG_PWM0_MODE,            REG_PWM0_MODE_MASK,         BIT2,           PINMUX_FOR_PWM0_MODE_4},
    {PAD_GPIO0,     CHIPTOP_BANK, REG_PWM8_MODE,            REG_PWM8_MODE_MASK,         BIT9,           PINMUX_FOR_PWM8_MODE_1},
    {PAD_GPIO0,     CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT10,          PINMUX_FOR_I2S_MODE_1},
    {PAD_GPIO0,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_GPIO0,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_GPIO1,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_GPIO1,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_GPIO1,     CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT6,           PINMUX_FOR_SPI1_MODE_4},
    {PAD_GPIO1,     CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT1,           PINMUX_FOR_FUART_MODE_2},
    {PAD_GPIO1,     CHIPTOP_BANK, REG_PWM1_MODE,            REG_PWM1_MODE_MASK,         BIT5,           PINMUX_FOR_PWM1_MODE_4},
    {PAD_GPIO1,     CHIPTOP_BANK, REG_PWM9_MODE,            REG_PWM9_MODE_MASK,         BIT12,          PINMUX_FOR_PWM9_MODE_1},
    {PAD_GPIO1,     CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT10,          PINMUX_FOR_I2S_MODE_1},
    {PAD_GPIO1,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_GPIO1,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_GPIO2,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_GPIO2,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_GPIO2,     CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT6,           PINMUX_FOR_SPI1_MODE_4},
    {PAD_GPIO2,     CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT1,           PINMUX_FOR_FUART_MODE_2},
    {PAD_GPIO2,     CHIPTOP_BANK, REG_PWM2_MODE,            REG_PWM2_MODE_MASK,         BIT8,           PINMUX_FOR_PWM2_MODE_4},
    {PAD_GPIO2,     CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT10,          PINMUX_FOR_I2S_MODE_1},
    {PAD_GPIO2,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_GPIO2,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_GPIO3,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_GPIO3,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_GPIO3,     CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT6,           PINMUX_FOR_SPI1_MODE_4},
    {PAD_GPIO3,     CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT1,           PINMUX_FOR_FUART_MODE_2},
    {PAD_GPIO3,     CHIPTOP_BANK, REG_PWM3_MODE,            REG_PWM3_MODE_MASK,         BIT11,          PINMUX_FOR_PWM3_MODE_4},
    {PAD_GPIO3,     CHIPTOP_BANK, REG_PWM10_MODE,           REG_PWM10_MODE_MASK,        BIT0,           PINMUX_FOR_PWM10_MODE_1},
    {PAD_GPIO3,     CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT10,          PINMUX_FOR_I2S_MODE_1},
    {PAD_GPIO3,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_GPIO3,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_GPIO4,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_GPIO4,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_GPIO4,     CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT1,           PINMUX_FOR_SPI0_MODE_2},
    {PAD_GPIO4,     CHIPTOP_BANK, REG_UART0_MODE,           REG_UART0_MODE_MASK,        BIT5|BIT4,      PINMUX_FOR_UART0_MODE_3},
    {PAD_GPIO4,     CHIPTOP_BANK, REG_PWM4_MODE,            REG_PWM4_MODE_MASK,         BIT13|BIT12,    PINMUX_FOR_PWM4_MODE_3},
    {PAD_GPIO4,     CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,          BIT2,           PINMUX_FOR_ETH_MODE},
    {PAD_GPIO4,     CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,         BIT8,           PINMUX_FOR_DMIC_MODE_1},
    {PAD_GPIO4,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_GPIO4,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_GPIO5,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_GPIO5,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_GPIO5,     CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT1,           PINMUX_FOR_SPI0_MODE_2},
    {PAD_GPIO5,     CHIPTOP_BANK, REG_UART0_MODE,           REG_UART0_MODE_MASK,        BIT5|BIT4,      PINMUX_FOR_UART0_MODE_3},
    {PAD_GPIO5,     CHIPTOP_BANK, REG_PWM5_MODE,            REG_PWM5_MODE_MASK,         BIT1|BIT0,      PINMUX_FOR_PWM5_MODE_3},
    {PAD_GPIO5,     CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,          BIT2,           PINMUX_FOR_ETH_MODE},
    {PAD_GPIO5,     CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,         BIT8,           PINMUX_FOR_DMIC_MODE_1},
    {PAD_GPIO5,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_GPIO5,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_GPIO6,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_GPIO6,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_GPIO6,     CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT1,           PINMUX_FOR_SPI0_MODE_2},
    {PAD_GPIO6,     CHIPTOP_BANK, REG_UART1_MODE,           REG_UART1_MODE_MASK,        BIT9|BIT8,      PINMUX_FOR_UART1_MODE_3},
    {PAD_GPIO6,     CHIPTOP_BANK, REG_PWM6_MODE,            REG_PWM6_MODE_MASK,         BIT4|BIT3,      PINMUX_FOR_PWM6_MODE_3},
    {PAD_GPIO6,     CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,          BIT2,           PINMUX_FOR_ETH_MODE},
    {PAD_GPIO6,     CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,         BIT8,           PINMUX_FOR_DMIC_MODE_1},
    {PAD_GPIO6,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_GPIO6,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_GPIO7,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_GPIO7,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_GPIO7,     CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT1,           PINMUX_FOR_SPI0_MODE_2},
    {PAD_GPIO7,     CHIPTOP_BANK, REG_UART1_MODE,           REG_UART1_MODE_MASK,        BIT9|BIT8,      PINMUX_FOR_UART1_MODE_3},
    {PAD_GPIO7,     CHIPTOP_BANK, REG_PWM7_MODE,            REG_PWM7_MODE_MASK,         BIT7|BIT6,      PINMUX_FOR_PWM7_MODE_3},
    {PAD_GPIO7,     CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,          BIT2,           PINMUX_FOR_ETH_MODE},
    {PAD_GPIO7,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_GPIO7,     CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_GPIO8,     CHIPTOP_BANK, REG_PWM8_MODE,            REG_PWM8_MODE_MASK,         BIT10,          PINMUX_FOR_PWM8_MODE_2},
    {PAD_GPIO8,     CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,          BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO9,     CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,          BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO12,    CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT5,           PINMUX_FOR_SPI1_MODE_2},
    {PAD_GPIO12,    CHIPTOP_BANK, REG_UART1_MODE,           REG_UART1_MODE_MASK,        BIT10,          PINMUX_FOR_UART1_MODE_4},
    {PAD_GPIO12,    CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,          BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO13,    CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT5,           PINMUX_FOR_SPI1_MODE_2},
    {PAD_GPIO13,    CHIPTOP_BANK, REG_UART1_MODE,           REG_UART1_MODE_MASK,        BIT10,          PINMUX_FOR_UART1_MODE_4},
    {PAD_GPIO13,    CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,          BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO14,    CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT5,           PINMUX_FOR_SPI1_MODE_2},
    {PAD_GPIO14,    CHIPTOP_BANK, REG_PWM0_MODE,            REG_PWM0_MODE_MASK,         BIT1,           PINMUX_FOR_PWM0_MODE_2},
    {PAD_GPIO14,    CHIPTOP_BANK, REG_PWM2_MODE,            REG_PWM2_MODE_MASK,         BIT6,           PINMUX_FOR_PWM2_MODE_1},
    {PAD_GPIO14,    CHIPTOP_BANK, REG_PWM9_MODE,            REG_PWM9_MODE_MASK,         BIT13|BIT12,    PINMUX_FOR_PWM9_MODE_3},
    {PAD_GPIO14,    CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,          BIT2,           PINMUX_FOR_ETH_MODE},
    {PAD_GPIO14,    CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_GPIO14,    CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_GPIO15,    CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT5,           PINMUX_FOR_SPI1_MODE_2},
    {PAD_GPIO15,    CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT1|BIT0,      PINMUX_FOR_FUART_MODE_3},
    {PAD_GPIO15,    CHIPTOP_BANK, REG_PWM1_MODE,            REG_PWM1_MODE_MASK,         BIT4,           PINMUX_FOR_PWM1_MODE_2},
    {PAD_GPIO15,    CHIPTOP_BANK, REG_PWM3_MODE,            REG_PWM3_MODE_MASK,         BIT9,           PINMUX_FOR_PWM3_MODE_1},
    {PAD_GPIO15,    CHIPTOP_BANK, REG_PWM10_MODE,           REG_PWM10_MODE_MASK,        BIT1|BIT0,      PINMUX_FOR_PWM10_MODE_3},
    {PAD_GPIO15,    CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,          BIT2,           PINMUX_FOR_ETH_MODE},
    {PAD_GPIO15,    CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_GPIO15,    CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_FUART_RX,  CHIPTOP_BANK, REG_EJ_MODE,              REG_EJ_MODE_MASK,           BIT0,           PINMUX_FOR_EJ_MODE_1},
    {PAD_FUART_RX,  CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_FUART_RX,  CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_FUART_RX,  CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT1|BIT0,      PINMUX_FOR_SPI0_MODE_3},
    {PAD_FUART_RX,  CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT0,           PINMUX_FOR_FUART_MODE_1},
    {PAD_FUART_RX,  CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT1|BIT0,      PINMUX_FOR_FUART_MODE_3},
    {PAD_FUART_RX,  CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT2|BIT0,      PINMUX_FOR_FUART_MODE_5},
    {PAD_FUART_RX,  CHIPTOP_BANK, REG_UART0_MODE,           REG_UART0_MODE_MASK,        BIT5,           PINMUX_FOR_UART0_MODE_2},
    {PAD_FUART_RX,  CHIPTOP_BANK, REG_PWM0_MODE,            REG_PWM0_MODE_MASK,         BIT1|BIT0,      PINMUX_FOR_PWM0_MODE_3},
    {PAD_FUART_RX,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_FUART_RX,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_FUART_TX,  CHIPTOP_BANK, REG_EJ_MODE,              REG_EJ_MODE_MASK,           BIT0,           PINMUX_FOR_EJ_MODE_1},
    {PAD_FUART_TX,  CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_FUART_TX,  CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_FUART_TX,  CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT1|BIT0,      PINMUX_FOR_SPI0_MODE_3},
    {PAD_FUART_TX,  CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT0,           PINMUX_FOR_FUART_MODE_1},
    {PAD_FUART_TX,  CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT1|BIT0,      PINMUX_FOR_FUART_MODE_3},
    {PAD_FUART_TX,  CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT2|BIT0,      PINMUX_FOR_FUART_MODE_5},
    {PAD_FUART_TX,  CHIPTOP_BANK, REG_UART0_MODE,           REG_UART0_MODE_MASK,        BIT5,           PINMUX_FOR_UART0_MODE_2},
    {PAD_FUART_TX,  CHIPTOP_BANK, REG_PWM1_MODE,            REG_PWM1_MODE_MASK,         BIT4|BIT3,      PINMUX_FOR_PWM1_MODE_3},
    {PAD_FUART_TX,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_FUART_TX,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_FUART_CTS, CHIPTOP_BANK, REG_EJ_MODE,              REG_EJ_MODE_MASK,           BIT0,           PINMUX_FOR_EJ_MODE_1},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT1|BIT0,      PINMUX_FOR_SPI0_MODE_3},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT0,           PINMUX_FOR_FUART_MODE_1},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT1|BIT0,      PINMUX_FOR_FUART_MODE_3},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_UART1_MODE,           REG_UART1_MODE_MASK,        BIT9,           PINMUX_FOR_UART1_MODE_2},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_PWM2_MODE,            REG_PWM2_MODE_MASK,         BIT7,           PINMUX_FOR_PWM2_MODE_2},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_FUART_RTS, CHIPTOP_BANK, REG_EJ_MODE,              REG_EJ_MODE_MASK,           BIT0,           PINMUX_FOR_EJ_MODE_1},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT1|BIT0,      PINMUX_FOR_SPI0_MODE_3},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT0,           PINMUX_FOR_FUART_MODE_1},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_UART1_MODE,           REG_UART1_MODE_MASK,        BIT9,           PINMUX_FOR_UART1_MODE_2},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_PWM3_MODE,            REG_PWM3_MODE_MASK,         BIT10,          PINMUX_FOR_PWM3_MODE_2},

    {PAD_I2C0_SCL,  CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_I2C0_SCL,  CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_I2C0_SCL,  CHIPTOP_BANK, REG_I2C0_MODE,            REG_I2C0_MODE_MASK,         BIT0,           PINMUX_FOR_I2C0_MODE_1},
    {PAD_I2C0_SCL,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_I2C0_SCL,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_I2C0_SDA,  CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_I2C0_SDA,  CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_I2C0_SDA,  CHIPTOP_BANK, REG_I2C0_MODE,            REG_I2C0_MODE_MASK,         BIT0,           PINMUX_FOR_I2C0_MODE_1},
    {PAD_I2C0_SDA,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_I2C0_SDA,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_I2C1_SCL,  CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_I2C1_SCL,  CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_I2C1_SCL,  CHIPTOP_BANK, REG_I2C1_MODE,            REG_I2C1_MODE_MASK,         BIT4,           PINMUX_FOR_I2C1_MODE_1},

    {PAD_I2C1_SDA,  CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_I2C1_SDA,  CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_I2C1_SDA,  CHIPTOP_BANK, REG_I2C1_MODE,            REG_I2C1_MODE_MASK,         BIT4,           PINMUX_FOR_I2C1_MODE_1},

    {PAD_SR_IO00,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SR_IO00,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SR_IO00,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO00,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO00,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},

    {PAD_SR_IO01,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT0,           PINMUX_FOR_MIPI_PAD_IN_1},
    {PAD_SR_IO01,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1|BIT0,      PINMUX_FOR_MIPI_PAD_IN_3},
    {PAD_SR_IO01,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SR_IO01,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SR_IO01,   CHIPTOP_BANK, REG_I2C0_MODE,            REG_I2C0_MODE_MASK,         BIT1|BIT0,      PINMUX_FOR_I2C0_MODE_3},
    {PAD_SR_IO01,   CHIPTOP_BANK, REG_I2C1_MODE,            REG_I2C1_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_I2C1_MODE_3},
    {PAD_SR_IO01,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO01,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO01,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},

    {PAD_SR_IO02,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT0,           PINMUX_FOR_MIPI_PAD_IN_1},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1|BIT0,      PINMUX_FOR_MIPI_PAD_IN_3},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_I2C0_MODE,            REG_I2C0_MODE_MASK,         BIT1|BIT0,      PINMUX_FOR_I2C0_MODE_3},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_I2C1_MODE,            REG_I2C1_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_I2C1_MODE_3},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},
    {PAD_SR_IO02,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO03,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT0,           PINMUX_FOR_MIPI_PAD_IN_1},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1,           PINMUX_FOR_MIPI_PAD_IN_2},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1|BIT0,      PINMUX_FOR_MIPI_PAD_IN_3},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},
    {PAD_SR_IO03,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO04,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT0,           PINMUX_FOR_MIPI_PAD_IN_1},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1,           PINMUX_FOR_MIPI_PAD_IN_2},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1|BIT0,      PINMUX_FOR_MIPI_PAD_IN_3},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},
    {PAD_SR_IO04,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO05,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT0,           PINMUX_FOR_MIPI_PAD_IN_1},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1,           PINMUX_FOR_MIPI_PAD_IN_2},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1|BIT0,      PINMUX_FOR_MIPI_PAD_IN_3},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},
    {PAD_SR_IO05,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO06,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT0,           PINMUX_FOR_MIPI_PAD_IN_1},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1,           PINMUX_FOR_MIPI_PAD_IN_2},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1|BIT0,      PINMUX_FOR_MIPI_PAD_IN_3},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},
    {PAD_SR_IO06,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO07,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT0,           PINMUX_FOR_MIPI_PAD_IN_1},
    {PAD_SR_IO07,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1,           PINMUX_FOR_MIPI_PAD_IN_2},
    {PAD_SR_IO07,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO07,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO07,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO07,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO07,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO07,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},
    {PAD_SR_IO07,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO07,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},
    {PAD_SR_IO07,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO08,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT0,           PINMUX_FOR_MIPI_PAD_IN_1},
    {PAD_SR_IO08,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT1,           PINMUX_FOR_MIPI_PAD_IN_2},
    {PAD_SR_IO08,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO08,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO08,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO08,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO08,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO08,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},
    {PAD_SR_IO08,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO08,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},
    {PAD_SR_IO08,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO09,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT0,           PINMUX_FOR_MIPI_PAD_IN_1},
    {PAD_SR_IO09,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO09,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO09,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO09,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO09,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO09,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},
    {PAD_SR_IO09,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO09,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},
    {PAD_SR_IO09,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO10,   CHIPTOP_BANK, REG_MIPI_PAD_IN,          REG_MIPI_PAD_IN_MASK,       BIT0,           PINMUX_FOR_MIPI_PAD_IN_1},
    {PAD_SR_IO10,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO10,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO10,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO10,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO10,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO10,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO10,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},

    {PAD_SR_IO11,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO11,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO11,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO11,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO11,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1|BIT0,      PINMUX_FOR_SR_MODE_3},
    {PAD_SR_IO11,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO11,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},

    {PAD_SR_IO12,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO12,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO12,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO12,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO12,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},
    {PAD_SR_IO12,   CHIPTOP_BANK, REG_SR_PDN_MODE,          REG_SR_PDN_MODE_MASK,       BIT9,           PINMUX_FOR_SR_PDN_MODE_1},
    {PAD_SR_IO12,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT4,           PINMUX_FOR_CCIR_MODE_1},
    {PAD_SR_IO12,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO13,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO13,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO13,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO13,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO13,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT2,           PINMUX_FOR_SR_MODE_4},
    {PAD_SR_IO13,   CHIPTOP_BANK, REG_SR_RST_MODE,          REG_SR_RST_MODE_MASK,       BIT11,          PINMUX_FOR_SR_RST_MODE_1},
    {PAD_SR_IO13,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO14,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO14,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO14,   CHIPTOP_BANK, REG_PWM8_MODE,            REG_PWM8_MODE_MASK,         BIT11,          PINMUX_FOR_PWM8_MODE_4},
    {PAD_SR_IO14,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO14,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO14,   CHIPTOP_BANK, REG_SR_HVSYNC_MODE,       REG_SR_HVSYNC_MODE_MASK,    BIT6,           PINMUX_FOR_SR_HVSYNC_MODE},

    {PAD_SR_IO15,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO15,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO15,   CHIPTOP_BANK, REG_PWM9_MODE,            REG_PWM9_MODE_MASK,         BIT14,          PINMUX_FOR_PWM9_MODE_4},
    {PAD_SR_IO15,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO15,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO15,   CHIPTOP_BANK, REG_SR_PCK_MODE,          REG_SR_PCK_MODE_MASK,       BIT8,           PINMUX_FOR_SR_PCK_MODE},
    {PAD_SR_IO15,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5,           PINMUX_FOR_CCIR_MODE_2},
    {PAD_SR_IO15,   CHIPTOP_BANK, REG_CCIR_MODE,            REG_CCIR_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_CCIR_MODE_3},

    {PAD_SR_IO16,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO16,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO16,   CHIPTOP_BANK, REG_PWM10_MODE,           REG_PWM10_MODE_MASK,        BIT2,           PINMUX_FOR_PWM10_MODE_4},
    {PAD_SR_IO16,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO16,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO16,   CHIPTOP_BANK, REG_SR_HVSYNC_MODE,       REG_SR_HVSYNC_MODE_MASK,    BIT6,           PINMUX_FOR_SR_HVSYNC_MODE},

    {PAD_SR_IO17,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SR_IO17,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SR_IO17,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT0,           PINMUX_FOR_SR_MODE_1},
    {PAD_SR_IO17,   CHIPTOP_BANK, REG_SR_MODE,              REG_SR_MODE_MASK,           BIT1,           PINMUX_FOR_SR_MODE_2},
    {PAD_SR_IO17,   CHIPTOP_BANK, REG_SR_MCLK_MODE,         REG_SR_MCLK_MODE_MASK,      BIT7,           PINMUX_FOR_SR_MCLK_MODE},

    {PAD_UART0_RX,  CHIPTOP_BANK, REG_UART0_MODE,           REG_UART0_MODE_MASK,        BIT4,           PINMUX_FOR_UART0_MODE_1},
    {PAD_UART0_RX,  CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,         BIT9|BIT8,      PINMUX_FOR_DMIC_MODE_3},

    {PAD_UART0_TX,  CHIPTOP_BANK, REG_UART0_MODE,           REG_UART0_MODE_MASK,        BIT4,           PINMUX_FOR_UART0_MODE_1},
    {PAD_UART0_TX,  CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,         BIT9|BIT8,      PINMUX_FOR_DMIC_MODE_3},

    {PAD_UART1_RX,  CHIPTOP_BANK, REG_I2C0_MODE,            REG_I2C0_MODE_MASK,         BIT2,           PINMUX_FOR_I2C0_MODE_4},
    {PAD_UART1_RX,  CHIPTOP_BANK, REG_UART1_MODE,           REG_UART1_MODE_MASK,        BIT8,           PINMUX_FOR_UART1_MODE_1},

    {PAD_UART1_TX,  CHIPTOP_BANK, REG_I2C0_MODE,            REG_I2C0_MODE_MASK,         BIT2,           PINMUX_FOR_I2C0_MODE_4},
    {PAD_UART1_TX,  CHIPTOP_BANK, REG_UART1_MODE,           REG_UART1_MODE_MASK,        BIT8,           PINMUX_FOR_UART1_MODE_1},

    {PAD_SPI0_CZ,   CHIPTOP_BANK, REG_EJ_MODE,              REG_EJ_MODE_MASK,           BIT1,           PINMUX_FOR_EJ_MODE_2},
    {PAD_SPI0_CZ,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SPI0_CZ,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SPI0_CZ,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SPI0_CZ,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SPI0_CZ,   CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT0,           PINMUX_FOR_SPI0_MODE_1},
    {PAD_SPI0_CZ,   CHIPTOP_BANK, REG_PWM4_MODE,            REG_PWM4_MODE_MASK,         BIT13,          PINMUX_FOR_PWM4_MODE_2},

    {PAD_SPI0_CK,   CHIPTOP_BANK, REG_EJ_MODE,              REG_EJ_MODE_MASK,           BIT1,           PINMUX_FOR_EJ_MODE_2},
    {PAD_SPI0_CK,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SPI0_CK,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SPI0_CK,   CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT0,           PINMUX_FOR_SPI0_MODE_1},
    {PAD_SPI0_CK,   CHIPTOP_BANK, REG_PWM5_MODE,            REG_PWM5_MODE_MASK,         BIT1,           PINMUX_FOR_PWM5_MODE_2},

    {PAD_SPI0_DI,   CHIPTOP_BANK, REG_EJ_MODE,              REG_EJ_MODE_MASK,           BIT1,           PINMUX_FOR_EJ_MODE_2},
    {PAD_SPI0_DI,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SPI0_DI,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SPI0_DI,   CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT0,           PINMUX_FOR_SPI0_MODE_1},
    {PAD_SPI0_DI,   CHIPTOP_BANK, REG_PWM6_MODE,            REG_PWM6_MODE_MASK,         BIT4,           PINMUX_FOR_PWM6_MODE_2},

    {PAD_SPI0_DO,   CHIPTOP_BANK, REG_EJ_MODE,              REG_EJ_MODE_MASK,           BIT1,           PINMUX_FOR_EJ_MODE_2},
    {PAD_SPI0_DO,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT0,           PINMUX_FOR_TEST_IN_MODE_1},
    {PAD_SPI0_DO,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT4,           PINMUX_FOR_TEST_OUT_MODE_1},
    {PAD_SPI0_DO,   CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT0,           PINMUX_FOR_SPI0_MODE_1},
    {PAD_SPI0_DO,   CHIPTOP_BANK, REG_PWM7_MODE,            REG_PWM7_MODE_MASK,         BIT7,           PINMUX_FOR_PWM7_MODE_2},

    {PAD_SPI1_CZ,   CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT4,           PINMUX_FOR_SPI1_MODE_1},
    {PAD_SPI1_CZ,   CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT11|BIT10,    PINMUX_FOR_I2S_MODE_3},

    {PAD_SPI1_CK,   CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT4,           PINMUX_FOR_SPI1_MODE_1},
    {PAD_SPI1_CK,   CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT11|BIT10,    PINMUX_FOR_I2S_MODE_3},

    {PAD_SPI1_DI,   CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT4,           PINMUX_FOR_SPI1_MODE_1},
    {PAD_SPI1_DI,   CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT11|BIT10,    PINMUX_FOR_I2S_MODE_3},

    {PAD_SPI1_DO,   CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT4,           PINMUX_FOR_SPI1_MODE_1},
    {PAD_SPI1_DO,   CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT11|BIT10,    PINMUX_FOR_I2S_MODE_3},

    {PAD_PWM0,      CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_PWM0,      CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_PWM0,      CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_PWM0,      CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_PWM0,      CHIPTOP_BANK, REG_I2C0_MODE,            REG_I2C0_MODE_MASK,         BIT1,           PINMUX_FOR_I2C0_MODE_2},
    {PAD_PWM0,      CHIPTOP_BANK, REG_I2C1_MODE,            REG_I2C1_MODE_MASK,         BIT5,           PINMUX_FOR_I2C1_MODE_2},
    {PAD_PWM0,      CHIPTOP_BANK, REG_PWM0_MODE,            REG_PWM0_MODE_MASK,         BIT0,           PINMUX_FOR_PWM0_MODE_1},
    {PAD_PWM0,      CHIPTOP_BANK, REG_PWM9_MODE,            REG_PWM9_MODE_MASK,         BIT13,          PINMUX_FOR_PWM9_MODE_2},
    {PAD_PWM0,      CHIPTOP_BANK, REG_SR_PDN_MODE,          REG_SR_PDN_MODE_MASK,       BIT10,          PINMUX_FOR_SR_PDN_MODE_2},

    {PAD_PWM1,      CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_PWM1,      CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_PWM1,      CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_PWM1,      CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_PWM1,      CHIPTOP_BANK, REG_I2C0_MODE,            REG_I2C0_MODE_MASK,         BIT1,           PINMUX_FOR_I2C0_MODE_2},
    {PAD_PWM1,      CHIPTOP_BANK, REG_I2C1_MODE,            REG_I2C1_MODE_MASK,         BIT5,           PINMUX_FOR_I2C1_MODE_2},
    {PAD_PWM1,      CHIPTOP_BANK, REG_PWM1_MODE,            REG_PWM1_MODE_MASK,         BIT3,           PINMUX_FOR_PWM1_MODE_1},
    {PAD_PWM1,      CHIPTOP_BANK, REG_PWM10_MODE,           REG_PWM10_MODE_MASK,        BIT1,           PINMUX_FOR_PWM10_MODE_2},
    {PAD_PWM1,      CHIPTOP_BANK, REG_SR_RST_MODE,          REG_SR_RST_MODE_MASK,       BIT12,          PINMUX_FOR_SR_RST_MODE_2},

    {PAD_SD_CLK,    CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_SD_CLK,    CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SD_CLK,    CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_SD_CLK,    CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SD_CLK,    CHIPTOP_BANK, REG_SD_MODE,              REG_SD_MODE_MASK,           BIT2,           PINMUX_FOR_SD_MODE},

    {PAD_SD_CMD,    CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_SD_CMD,    CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SD_CMD,    CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_SD_CMD,    CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SD_CMD,    CHIPTOP_BANK, REG_SD_MODE,              REG_SD_MODE_MASK,           BIT2,           PINMUX_FOR_SD_MODE},

    {PAD_SD_D0,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_SD_D0,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_SD_D0,     CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_SPI1_MODE_3},
    {PAD_SD_D0,     CHIPTOP_BANK, REG_SD_MODE,              REG_SD_MODE_MASK,           BIT2,           PINMUX_FOR_SD_MODE},

    {PAD_SD_D1,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_SD_D1,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_SD_D1,     CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_SPI1_MODE_3},
    {PAD_SD_D1,     CHIPTOP_BANK, REG_SD_MODE,              REG_SD_MODE_MASK,           BIT2,           PINMUX_FOR_SD_MODE},

    {PAD_SD_D2,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_SD_D2,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_SD_D2,     CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_SPI1_MODE_3},
    {PAD_SD_D2,     CHIPTOP_BANK, REG_SD_MODE,              REG_SD_MODE_MASK,           BIT2,           PINMUX_FOR_SD_MODE},

    {PAD_SD_D3,     CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_SD_D3,     CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_SD_D3,     CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,         BIT5|BIT4,      PINMUX_FOR_SPI1_MODE_3},
    {PAD_SD_D3,     CHIPTOP_BANK, REG_SD_MODE,              REG_SD_MODE_MASK,           BIT2,           PINMUX_FOR_SD_MODE},

    {PAD_PM_SD_CDZ, CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_PM_SD_CDZ, CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_PM_SD_CDZ, PMSLEEP_BANK, REG_PM_SD_CDZ_MODE,       REG_PM_SD_CDZ_MODE_MASK,    BIT14,          PINMUX_FOR_PM_SD_CDZ_MODE},

    {PAD_PM_IRIN,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_PM_IRIN,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_PM_IRIN,   PMSLEEP_BANK, REG_PM_IR_IS_GPIO,        REG_PM_IR_IS_GPIO_MASK,     0,              PINMUX_FOR_PM_IRIN_MODE},

    {PAD_PM_GPIO0,  PMSLEEP_BANK, REG_PM_PWM0_MODE,         REG_PM_PWM0_MODE_MASK,      BIT0,           PINMUX_FOR_PM_PWM0_MODE_1},
    {PAD_PM_GPIO0,  PMSLEEP_BANK, REG_PM_VID_MODE,          REG_PM_VID_MODE_MASK,       BIT13,          PINMUX_FOR_PM_VID_MODE_2},
    {PAD_PM_GPIO0,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_PM_GPIO0,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_PM_GPIO1,  PMSLEEP_BANK, REG_PM_PWM1_MODE,         REG_PM_PWM1_MODE_MASK,      BIT2,           PINMUX_FOR_PM_PWM1_MODE_1},
    {PAD_PM_GPIO1,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_PM_GPIO1,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    {PAD_PM_GPIO2,  PMSLEEP_BANK, REG_PM_PWM2_MODE,         REG_PM_PWM2_MODE_MASK,      BIT6,           PINMUX_FOR_PM_PWM2_MODE_1},

    {PAD_PM_GPIO3,  PMSLEEP_BANK, REG_PM_PWM3_MODE,         REG_PM_PWM3_MODE_MASK,      BIT8,           PINMUX_FOR_PM_PWM3_MODE_1},
    {PAD_PM_GPIO3,  PMSLEEP_BANK, REG_PM_UART1_MODE,        REG_PM_UART1_MODE_MASK,     BIT8,           PINMUX_FOR_PM_UART1_MODE},
    {PAD_PM_GPIO3,  PMSLEEP_BANK, REG_PM_VID_MODE,          REG_PM_VID_MODE_MASK,       BIT13,          PINMUX_FOR_PM_VID_MODE_2},
    {PAD_PM_GPIO3,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_PM_GPIO3,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},

    /*
    {PAD_PM_GPIO4,  PMSLEEP_BANK, REG_PM_UART1_MODE,        REG_PM_UART1_MODE_MASK,     BIT8,           PINMUX_FOR_PM_UART1_MODE},
    {PAD_PM_GPIO4,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},
    {PAD_PM_GPIO4,  CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT7,           PINMUX_FOR_TTL_MODE_2},
    */

    {PAD_PM_GPIO7,  PMSLEEP_BANK, REG_PM_PWM3_MODE,         REG_PM_PWM3_MODE_MASK,      BIT9,           PINMUX_FOR_PM_PWM3_MODE_2},

    {PAD_PM_GPIO9,  PMSLEEP_BANK, REG_PM_SPICSZ2_GPIO,      REG_PM_SPICSZ2_GPIO_MASK,   0,              PINMUX_FOR_PM_SPICSZ2_MODE},
    {PAD_PM_GPIO9,  PMSLEEP_BANK, REG_PM_PWM2_MODE,         REG_PM_PWM2_MODE_MASK,      BIT7,           PINMUX_FOR_PM_PWM2_MODE_2},
    {PAD_PM_GPIO9,  PMSLEEP_BANK, REG_PM_PWM8_MODE,         REG_PM_PWM8_MODE_MASK,      BIT2,           PINMUX_FOR_PM_PWM8_MODE},

    {PAD_PM_SPI_CZ, CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_PM_SPI_CZ, CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_PM_SPI_CZ, PMSLEEP_BANK, REG_PM_SPICSZ1_GPIO,      REG_PM_SPICSZ1_GPIO_MASK,   0,              PINMUX_FOR_PM_SPICSZ1_MODE},

    {PAD_PM_SPI_CK, CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_PM_SPI_CK, CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_PM_SPI_CK, PMSLEEP_BANK, REG_PM_SPI_GPIO,          REG_PM_SPI_GPIO_MASK,       0,              PINMUX_FOR_PM_SPI_MODE},

    {PAD_PM_SPI_DI, CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_PM_SPI_DI, CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_PM_SPI_DI, PMSLEEP_BANK, REG_PM_SPI_GPIO,          REG_PM_SPI_GPIO_MASK,       0,              PINMUX_FOR_PM_SPI_MODE},

    {PAD_PM_SPI_DO, CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_PM_SPI_DO, CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_PM_SPI_DO, PMSLEEP_BANK, REG_PM_SPI_GPIO,          REG_PM_SPI_GPIO_MASK,       0,              PINMUX_FOR_PM_SPI_MODE},

    {PAD_PM_SPI_WPZ,CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_PM_SPI_WPZ,CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    {PAD_PM_SPI_WPZ,PMSLEEP_BANK, REG_PM_SPIWPN_GPIO,       REG_PM_SPIWPN_GPIO_MASK,    0,              PINMUX_FOR_PM_SPIWPN_MODE},

    {PAD_PM_SPI_HLD,PMSLEEP_BANK, REG_PM_SPIHOLDN_MODE,     REG_PM_SPIHOLDN_MODE_MASK,  0,              PINMUX_FOR_PM_SPIHOLDN_MODE},

    {PAD_PM_LED0,   PMSLEEP_BANK, REG_PM_PWM9_MODE,         REG_PM_PWM9_MODE_MASK,      BIT3,           PINMUX_FOR_PM_PWM9_MODE},
    {PAD_PM_LED0,   PMSLEEP_BANK, REG_PM_VID_MODE,          REG_PM_VID_MODE_MASK,       BIT12,          PINMUX_FOR_PM_VID_MODE_1},
    {PAD_PM_LED0,   PMSLEEP_BANK, REG_PM_VID_MODE,          REG_PM_VID_MODE_MASK,       BIT13|BIT12,    PINMUX_FOR_PM_VID_MODE_3},
    {PAD_PM_LED0,   PMSLEEP_BANK, REG_PM_LED_MODE,          REG_PM_LED_MODE_MASK,       BIT4,           PINMUX_FOR_PM_LED_MODE},

    {PAD_PM_LED1,   PMSLEEP_BANK, REG_PM_PWM5_MODE,         REG_PM_PWM5_MODE_MASK,      BIT1,           PINMUX_FOR_PM_PWM5_MODE},
    {PAD_PM_LED1,   PMSLEEP_BANK, REG_PM_PWM10_MODE,        REG_PM_PWM10_MODE_MASK,     BIT4,           PINMUX_FOR_PM_PWM10_MODE},
    {PAD_PM_LED1,   PMSLEEP_BANK, REG_PM_VID_MODE,          REG_PM_VID_MODE_MASK,       BIT12,          PINMUX_FOR_PM_VID_MODE_1},
    {PAD_PM_LED1,   PMSLEEP_BANK, REG_PM_VID_MODE,          REG_PM_VID_MODE_MASK,       BIT13|BIT12,    PINMUX_FOR_PM_VID_MODE_3},
    {PAD_PM_LED1,   PMSLEEP_BANK, REG_PM_LED_MODE,          REG_PM_LED_MODE_MASK,       BIT4,           PINMUX_FOR_PM_LED_MODE},

    /*
    {PAD_SAR_GPIO0, CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_SAR_GPIO0, CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},

    {PAD_ETH_RN,    CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_ETH_RN,    CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},

    {PAD_USB_DM,    CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1,           PINMUX_FOR_TEST_IN_MODE_2},
    {PAD_USB_DM,    CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5,           PINMUX_FOR_TEST_OUT_MODE_2},
    */

    {PAD_SD1_IO0,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SD1_IO0,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SD1_IO0,   CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT2,           PINMUX_FOR_SPI0_MODE_4},
    {PAD_SD1_IO0,   CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT2,           PINMUX_FOR_FUART_MODE_4},
    {PAD_SD1_IO0,   CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT2|BIT1,      PINMUX_FOR_FUART_MODE_6},
    {PAD_SD1_IO0,   CHIPTOP_BANK, REG_SDIO_MODE,            REG_SDIO_MODE_MASK,         BIT8,           PINMUX_FOR_SDIO_MODE},
    {PAD_SD1_IO0,   CHIPTOP_BANK, REG_PWM0_MODE,            REG_PWM0_MODE_MASK,         BIT2|BIT0,      PINMUX_FOR_PWM0_MODE_5},
    {PAD_SD1_IO0,   CHIPTOP_BANK, REG_PWM2_MODE,            REG_PWM2_MODE_MASK,         BIT7|BIT6,      PINMUX_FOR_PWM2_MODE_3},
    {PAD_SD1_IO0,   CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT11,          PINMUX_FOR_I2S_MODE_2},
    {PAD_SD1_IO0,   CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},

    {PAD_SD1_IO1,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SD1_IO1,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SD1_IO1,   CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT2,           PINMUX_FOR_SPI0_MODE_4},
    {PAD_SD1_IO1,   CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT2,           PINMUX_FOR_FUART_MODE_4},
    {PAD_SD1_IO1,   CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT2|BIT1,      PINMUX_FOR_FUART_MODE_6},
    {PAD_SD1_IO1,   CHIPTOP_BANK, REG_SDIO_MODE,            REG_SDIO_MODE_MASK,         BIT8,           PINMUX_FOR_SDIO_MODE},
    {PAD_SD1_IO1,   CHIPTOP_BANK, REG_PWM1_MODE,            REG_PWM1_MODE_MASK,         BIT5|BIT3,      PINMUX_FOR_PWM1_MODE_5},
    {PAD_SD1_IO1,   CHIPTOP_BANK, REG_PWM3_MODE,            REG_PWM3_MODE_MASK,         BIT10|BIT9,     PINMUX_FOR_PWM3_MODE_3},
    {PAD_SD1_IO1,   CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT11,          PINMUX_FOR_I2S_MODE_2},
    {PAD_SD1_IO1,   CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},

    {PAD_SD1_IO2,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SD1_IO2,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SD1_IO2,   CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT2,           PINMUX_FOR_SPI0_MODE_4},
    {PAD_SD1_IO2,   CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT2,           PINMUX_FOR_FUART_MODE_4},
    {PAD_SD1_IO2,   CHIPTOP_BANK, REG_SDIO_MODE,            REG_SDIO_MODE_MASK,         BIT8,           PINMUX_FOR_SDIO_MODE},
    {PAD_SD1_IO2,   CHIPTOP_BANK, REG_PWM2_MODE,            REG_PWM2_MODE_MASK,         BIT8|BIT6,      PINMUX_FOR_PWM2_MODE_5},
    {PAD_SD1_IO2,   CHIPTOP_BANK, REG_PWM4_MODE,            REG_PWM4_MODE_MASK,         BIT12,          PINMUX_FOR_PWM4_MODE_1},
    {PAD_SD1_IO2,   CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT11,          PINMUX_FOR_I2S_MODE_2},
    {PAD_SD1_IO2,   CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},

    {PAD_SD1_IO3,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SD1_IO3,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SD1_IO3,   CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,         BIT2,           PINMUX_FOR_SPI0_MODE_4},
    {PAD_SD1_IO3,   CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,        BIT2,           PINMUX_FOR_FUART_MODE_4},
    {PAD_SD1_IO3,   CHIPTOP_BANK, REG_SDIO_MODE,            REG_SDIO_MODE_MASK,         BIT8,           PINMUX_FOR_SDIO_MODE},
    {PAD_SD1_IO3,   CHIPTOP_BANK, REG_PWM3_MODE,            REG_PWM3_MODE_MASK,         BIT11|BIT9,     PINMUX_FOR_PWM3_MODE_5},
    {PAD_SD1_IO3,   CHIPTOP_BANK, REG_PWM5_MODE,            REG_PWM5_MODE_MASK,         BIT0,           PINMUX_FOR_PWM5_MODE_1},
    {PAD_SD1_IO3,   CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,          BIT11,          PINMUX_FOR_I2S_MODE_2},
    {PAD_SD1_IO3,   CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},

    {PAD_SD1_IO4,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SD1_IO4,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SD1_IO4,   CHIPTOP_BANK, REG_SDIO_MODE,            REG_SDIO_MODE_MASK,         BIT8,           PINMUX_FOR_SDIO_MODE},
    {PAD_SD1_IO4,   CHIPTOP_BANK, REG_PWM4_MODE,            REG_PWM4_MODE_MASK,         BIT14,          PINMUX_FOR_PWM4_MODE_4},
    {PAD_SD1_IO4,   CHIPTOP_BANK, REG_PWM6_MODE,            REG_PWM6_MODE_MASK,         BIT3,           PINMUX_FOR_PWM6_MODE_1},
    {PAD_SD1_IO4,   CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,         BIT9,           PINMUX_FOR_DMIC_MODE_2},
    {PAD_SD1_IO4,   CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},

    {PAD_SD1_IO5,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SD1_IO5,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SD1_IO5,   CHIPTOP_BANK, REG_SDIO_MODE,            REG_SDIO_MODE_MASK,         BIT8,           PINMUX_FOR_SDIO_MODE},
    {PAD_SD1_IO5,   CHIPTOP_BANK, REG_PWM5_MODE,            REG_PWM5_MODE_MASK,         BIT2,           PINMUX_FOR_PWM5_MODE_4},
    {PAD_SD1_IO5,   CHIPTOP_BANK, REG_PWM7_MODE,            REG_PWM7_MODE_MASK,         BIT6,           PINMUX_FOR_PWM7_MODE_1},
    {PAD_SD1_IO5,   CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,         BIT9,           PINMUX_FOR_DMIC_MODE_2},
    {PAD_SD1_IO5,   CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},

    {PAD_SD1_IO6,   CHIPTOP_BANK, REG_TEST_IN_MODE,         REG_TEST_IN_MODE_MASK,      BIT1|BIT0,      PINMUX_FOR_TEST_IN_MODE_3},
    {PAD_SD1_IO6,   CHIPTOP_BANK, REG_TEST_OUT_MODE,        REG_TEST_OUT_MODE_MASK,     BIT5|BIT4,      PINMUX_FOR_TEST_OUT_MODE_3},
    {PAD_SD1_IO6,   CHIPTOP_BANK, REG_UART0_MODE,           REG_UART0_MODE_MASK,        BIT6,           PINMUX_FOR_UART0_MODE_4},
    {PAD_SD1_IO6,   CHIPTOP_BANK, REG_PWM6_MODE,            REG_PWM6_MODE_MASK,         BIT5,           PINMUX_FOR_PWM6_MODE_4},
    {PAD_SD1_IO6,   CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,         BIT9,           PINMUX_FOR_DMIC_MODE_2},
    {PAD_SD1_IO6,   CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},

    {PAD_SD1_IO7,   CHIPTOP_BANK, REG_UART0_MODE,           REG_UART0_MODE_MASK,        BIT6,           PINMUX_FOR_UART0_MODE_4},
    {PAD_SD1_IO7,   CHIPTOP_BANK, REG_PWM7_MODE,            REG_PWM7_MODE_MASK,         BIT8,           PINMUX_FOR_PWM7_MODE_4},
    {PAD_SD1_IO7,   CHIPTOP_BANK, REG_TTL_MODE,             REG_TTL_MODE_MASK,          BIT6,           PINMUX_FOR_TTL_MODE_1},

    {PAD_SD1_IO8,   CHIPTOP_BANK, REG_PWM8_MODE,            REG_PWM8_MODE_MASK,         BIT10|BIT9,     PINMUX_FOR_PWM8_MODE_3},
    {PAD_SD1_IO8,   CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,         BIT9|BIT8,      PINMUX_FOR_DMIC_MODE_3},

};

static const ST_PadModeInfo m_stPadModeInfoTbl[] =
{
    {"GPIO",    0,                                              0},
    // Non PM
    {"EJ_MODE_1",       _RIUA_16BIT(CHIPTOP_BANK,REG_EJ_MODE),           REG_EJ_MODE_MASK},
    {"EJ_MODE_2",       _RIUA_16BIT(CHIPTOP_BANK,REG_EJ_MODE),           REG_EJ_MODE_MASK},
    {"ALLPAD_IN",       _RIUA_16BIT(CHIPTOP_BANK,REG_ALLPAD_IN),         REG_ALLPAD_IN_MASK},
    {"MIPI_PAD_IN_1",   _RIUA_16BIT(CHIPTOP_BANK,REG_MIPI_PAD_IN),       REG_MIPI_PAD_IN_MASK},
    {"MIPI_PAD_IN_2",   _RIUA_16BIT(CHIPTOP_BANK,REG_MIPI_PAD_IN),       REG_MIPI_PAD_IN_MASK},
    {"MIPI_PAD_IN_3",   _RIUA_16BIT(CHIPTOP_BANK,REG_MIPI_PAD_IN),       REG_MIPI_PAD_IN_MASK},
    {"TEST_IN_MODE_1",  _RIUA_16BIT(CHIPTOP_BANK,REG_TEST_IN_MODE),      REG_TEST_IN_MODE_MASK},
    {"TEST_IN_MODE_2",  _RIUA_16BIT(CHIPTOP_BANK,REG_TEST_IN_MODE),      REG_TEST_IN_MODE_MASK},
    {"TEST_IN_MODE_3",  _RIUA_16BIT(CHIPTOP_BANK,REG_TEST_IN_MODE),      REG_TEST_IN_MODE_MASK},
    {"TEST_OUT_MODE_1", _RIUA_16BIT(CHIPTOP_BANK,REG_TEST_OUT_MODE),     REG_TEST_OUT_MODE_MASK},
    {"TEST_OUT_MODE_2", _RIUA_16BIT(CHIPTOP_BANK,REG_TEST_OUT_MODE),     REG_TEST_OUT_MODE_MASK},
    {"TEST_OUT_MODE_3", _RIUA_16BIT(CHIPTOP_BANK,REG_TEST_OUT_MODE),     REG_TEST_OUT_MODE_MASK},
    {"I2C0_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_I2C0_MODE),         REG_I2C0_MODE_MASK},
    {"I2C0_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_I2C0_MODE),         REG_I2C0_MODE_MASK},
    {"I2C0_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_I2C0_MODE),         REG_I2C0_MODE_MASK},
    {"I2C0_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_I2C0_MODE),         REG_I2C0_MODE_MASK},
    {"I2C1_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_I2C1_MODE),         REG_I2C1_MODE_MASK},
    {"I2C1_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_I2C1_MODE),         REG_I2C1_MODE_MASK},
    {"I2C1_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_I2C1_MODE),         REG_I2C1_MODE_MASK},
    {"SPI0_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_SPI0_MODE),         REG_SPI0_MODE_MASK},
    {"SPI0_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_SPI0_MODE),         REG_SPI0_MODE_MASK},
    {"SPI0_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_SPI0_MODE),         REG_SPI0_MODE_MASK},
    {"SPI0_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_SPI0_MODE),         REG_SPI0_MODE_MASK},
    {"SPI1_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_SPI1_MODE),         REG_SPI1_MODE_MASK},
    {"SPI1_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_SPI1_MODE),         REG_SPI1_MODE_MASK},
    {"SPI1_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_SPI1_MODE),         REG_SPI1_MODE_MASK},
    {"SPI1_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_SPI1_MODE),         REG_SPI1_MODE_MASK},
    {"FUART_MODE_1",    _RIUA_16BIT(CHIPTOP_BANK,REG_FUART_MODE),        REG_FUART_MODE_MASK},
    {"FUART_MODE_2",    _RIUA_16BIT(CHIPTOP_BANK,REG_FUART_MODE),        REG_FUART_MODE_MASK},
    {"FUART_MODE_3",    _RIUA_16BIT(CHIPTOP_BANK,REG_FUART_MODE),        REG_FUART_MODE_MASK},
    {"FUART_MODE_4",    _RIUA_16BIT(CHIPTOP_BANK,REG_FUART_MODE),        REG_FUART_MODE_MASK},
    {"FUART_MODE_5",    _RIUA_16BIT(CHIPTOP_BANK,REG_FUART_MODE),        REG_FUART_MODE_MASK},
    {"FUART_MODE_6",    _RIUA_16BIT(CHIPTOP_BANK,REG_FUART_MODE),        REG_FUART_MODE_MASK},
    {"UART0_MODE_1",    _RIUA_16BIT(CHIPTOP_BANK,REG_UART0_MODE),        REG_UART0_MODE_MASK},
    {"UART0_MODE_2",    _RIUA_16BIT(CHIPTOP_BANK,REG_UART0_MODE),        REG_UART0_MODE_MASK},
    {"UART0_MODE_3",    _RIUA_16BIT(CHIPTOP_BANK,REG_UART0_MODE),        REG_UART0_MODE_MASK},
    {"UART0_MODE_4",    _RIUA_16BIT(CHIPTOP_BANK,REG_UART0_MODE),        REG_UART0_MODE_MASK},
    {"UART1_MODE_1",    _RIUA_16BIT(CHIPTOP_BANK,REG_UART1_MODE),        REG_UART1_MODE_MASK},
    {"UART1_MODE_2",    _RIUA_16BIT(CHIPTOP_BANK,REG_UART1_MODE),        REG_UART1_MODE_MASK},
    {"UART1_MODE_3",    _RIUA_16BIT(CHIPTOP_BANK,REG_UART1_MODE),        REG_UART1_MODE_MASK},
    {"UART1_MODE_4",    _RIUA_16BIT(CHIPTOP_BANK,REG_UART1_MODE),        REG_UART1_MODE_MASK},
    {"SD_MODE",         _RIUA_16BIT(CHIPTOP_BANK,REG_SD_MODE),           REG_SD_MODE_MASK},
    {"SDIO_MODE",       _RIUA_16BIT(CHIPTOP_BANK,REG_SDIO_MODE),         REG_SDIO_MODE_MASK},
    {"PWM0_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM0_MODE),         REG_PWM0_MODE_MASK},
    {"PWM0_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM0_MODE),         REG_PWM0_MODE_MASK},
    {"PWM0_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM0_MODE),         REG_PWM0_MODE_MASK},
    {"PWM0_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM0_MODE),         REG_PWM0_MODE_MASK},
    {"PWM0_MODE_5",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM0_MODE),         REG_PWM0_MODE_MASK},
    {"PWM1_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM1_MODE),         REG_PWM1_MODE_MASK},
    {"PWM1_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM1_MODE),         REG_PWM1_MODE_MASK},
    {"PWM1_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM1_MODE),         REG_PWM1_MODE_MASK},
    {"PWM1_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM1_MODE),         REG_PWM1_MODE_MASK},
    {"PWM1_MODE_5",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM1_MODE),         REG_PWM1_MODE_MASK},
    {"PWM2_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM2_MODE),         REG_PWM2_MODE_MASK},
    {"PWM2_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM2_MODE),         REG_PWM2_MODE_MASK},
    {"PWM2_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM2_MODE),         REG_PWM2_MODE_MASK},
    {"PWM2_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM2_MODE),         REG_PWM2_MODE_MASK},
    {"PWM2_MODE_5",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM2_MODE),         REG_PWM2_MODE_MASK},
    {"PWM3_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM3_MODE),         REG_PWM3_MODE_MASK},
    {"PWM3_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM3_MODE),         REG_PWM3_MODE_MASK},
    {"PWM3_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM3_MODE),         REG_PWM3_MODE_MASK},
    {"PWM3_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM3_MODE),         REG_PWM3_MODE_MASK},
    {"PWM3_MODE_5",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM3_MODE),         REG_PWM3_MODE_MASK},
    {"PWM4_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM4_MODE),         REG_PWM4_MODE_MASK},
    {"PWM4_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM4_MODE),         REG_PWM4_MODE_MASK},
    {"PWM4_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM4_MODE),         REG_PWM4_MODE_MASK},
    {"PWM4_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM4_MODE),         REG_PWM4_MODE_MASK},
    {"PWM5_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM5_MODE),         REG_PWM5_MODE_MASK},
    {"PWM5_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM5_MODE),         REG_PWM5_MODE_MASK},
    {"PWM5_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM5_MODE),         REG_PWM5_MODE_MASK},
    {"PWM5_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM5_MODE),         REG_PWM5_MODE_MASK},
    {"PWM6_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM6_MODE),         REG_PWM6_MODE_MASK},
    {"PWM6_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM6_MODE),         REG_PWM6_MODE_MASK},
    {"PWM6_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM6_MODE),         REG_PWM6_MODE_MASK},
    {"PWM6_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM6_MODE),         REG_PWM6_MODE_MASK},
    {"PWM7_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM7_MODE),         REG_PWM7_MODE_MASK},
    {"PWM7_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM7_MODE),         REG_PWM7_MODE_MASK},
    {"PWM7_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM7_MODE),         REG_PWM7_MODE_MASK},
    {"PWM7_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM7_MODE),         REG_PWM7_MODE_MASK},
    {"PWM8_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM8_MODE),         REG_PWM8_MODE_MASK},
    {"PWM8_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM8_MODE),         REG_PWM8_MODE_MASK},
    {"PWM8_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM8_MODE),         REG_PWM8_MODE_MASK},
    {"PWM8_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM8_MODE),         REG_PWM8_MODE_MASK},
    {"PWM9_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM9_MODE),         REG_PWM9_MODE_MASK},
    {"PWM9_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM9_MODE),         REG_PWM9_MODE_MASK},
    {"PWM9_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM9_MODE),         REG_PWM9_MODE_MASK},
    {"PWM9_MODE_4",     _RIUA_16BIT(CHIPTOP_BANK,REG_PWM9_MODE),         REG_PWM9_MODE_MASK},
    {"PWM10_MODE_1",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM10_MODE),        REG_PWM10_MODE_MASK},
    {"PWM10_MODE_2",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM10_MODE),        REG_PWM10_MODE_MASK},
    {"PWM10_MODE_3",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM10_MODE),        REG_PWM10_MODE_MASK},
    {"PWM10_MODE_4",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM10_MODE),        REG_PWM10_MODE_MASK},
    {"SR_MODE_1",       _RIUA_16BIT(CHIPTOP_BANK,REG_SR_MODE),           REG_SR_MODE_MASK},
    {"SR_MODE_2",       _RIUA_16BIT(CHIPTOP_BANK,REG_SR_MODE),           REG_SR_MODE_MASK},
    {"SR_MODE_3",       _RIUA_16BIT(CHIPTOP_BANK,REG_SR_MODE),           REG_SR_MODE_MASK},
    {"SR_MODE_4",       _RIUA_16BIT(CHIPTOP_BANK,REG_SR_MODE),           REG_SR_MODE_MASK},
    {"SR_MCLK_MODE",    _RIUA_16BIT(CHIPTOP_BANK,REG_SR_MCLK_MODE),      REG_SR_MCLK_MODE_MASK},
    {"SR_PDN_MODE_1",   _RIUA_16BIT(CHIPTOP_BANK,REG_SR_PDN_MODE),       REG_SR_PDN_MODE_MASK},
    {"SR_PDN_MODE_2",   _RIUA_16BIT(CHIPTOP_BANK,REG_SR_PDN_MODE),       REG_SR_PDN_MODE_MASK},
    {"SR_RST_MODE_1",   _RIUA_16BIT(CHIPTOP_BANK,REG_SR_RST_MODE),       REG_SR_RST_MODE_MASK},
    {"SR_RST_MODE_2",   _RIUA_16BIT(CHIPTOP_BANK,REG_SR_RST_MODE),       REG_SR_RST_MODE_MASK},
    {"SR_HVSYNC_MODE",  _RIUA_16BIT(CHIPTOP_BANK,REG_SR_HVSYNC_MODE),    REG_SR_HVSYNC_MODE_MASK},
    {"SR_PCK_MODE",     _RIUA_16BIT(CHIPTOP_BANK,REG_SR_PCK_MODE),       REG_SR_PCK_MODE_MASK},
    {"ETH_MODE",        _RIUA_16BIT(CHIPTOP_BANK,REG_ETH_MODE),          REG_ETH_MODE_MASK},
    {"I2S_MODE_1",      _RIUA_16BIT(CHIPTOP_BANK,REG_I2S_MODE),          REG_I2S_MODE_MASK},
    {"I2S_MODE_2",      _RIUA_16BIT(CHIPTOP_BANK,REG_I2S_MODE),          REG_I2S_MODE_MASK},
    {"I2S_MODE_3",      _RIUA_16BIT(CHIPTOP_BANK,REG_I2S_MODE),          REG_I2S_MODE_MASK},
    {"DMIC_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_DMIC_MODE),         REG_DMIC_MODE_MASK},
    {"DMIC_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_DMIC_MODE),         REG_DMIC_MODE_MASK},
    {"DMIC_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_DMIC_MODE),         REG_DMIC_MODE_MASK},
    {"TTL_MODE_1",      _RIUA_16BIT(CHIPTOP_BANK,REG_TTL_MODE),          REG_TTL_MODE_MASK},
    {"TTL_MODE_2",      _RIUA_16BIT(CHIPTOP_BANK,REG_TTL_MODE),          REG_TTL_MODE_MASK},
    {"CCIR_MODE_1",     _RIUA_16BIT(CHIPTOP_BANK,REG_CCIR_MODE),         REG_CCIR_MODE_MASK},
    {"CCIR_MODE_2",     _RIUA_16BIT(CHIPTOP_BANK,REG_CCIR_MODE),         REG_CCIR_MODE_MASK},
    {"CCIR_MODE_3",     _RIUA_16BIT(CHIPTOP_BANK,REG_CCIR_MODE),         REG_CCIR_MODE_MASK},
    // PM Sleep
    {"PM_SPI_MODE",     _RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_GPIO),       REG_PM_SPI_GPIO_MASK},
    {"PM_SPIWPN_MODE",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPIWPN_GPIO),    REG_PM_SPIWPN_GPIO_MASK},
    {"PM_SPIHOLDN_MODE",_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPIHOLDN_MODE),  REG_PM_SPIHOLDN_MODE_MASK},
    {"PM_SPICSZ1_MODE", _RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPICSZ1_GPIO),   REG_PM_SPICSZ1_GPIO_MASK},
    {"PM_SPICSZ2_MODE", _RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPICSZ2_GPIO),   REG_PM_SPICSZ2_GPIO_MASK},
    {"PM_PWM0_MODE_1",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM0_MODE),      REG_PM_PWM0_MODE_MASK},
    {"PM_PWM0_MODE_2",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM0_MODE),      REG_PM_PWM0_MODE_MASK},
    {"PM_PWM1_MODE_1",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM1_MODE),      REG_PM_PWM1_MODE_MASK},
    {"PM_PWM1_MODE_2",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM1_MODE),      REG_PM_PWM1_MODE_MASK},
    {"PM_PWM2_MODE_1",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM2_MODE),      REG_PM_PWM2_MODE_MASK},
    {"PM_PWM2_MODE_2",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM2_MODE),      REG_PM_PWM2_MODE_MASK},
    {"PM_PWM3_MODE_1",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM3_MODE),      REG_PM_PWM3_MODE_MASK},
    {"PM_PWM3_MODE_2",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM3_MODE),      REG_PM_PWM3_MODE_MASK},
    {"PM_PWM4_MODE",    _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM4_MODE),      REG_PM_PWM4_MODE_MASK},
    {"PM_PWM5_MODE",    _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM5_MODE),      REG_PM_PWM5_MODE_MASK},
    {"PM_PWM8_MODE",    _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM8_MODE),      REG_PM_PWM8_MODE_MASK},
    {"PM_PWM9_MODE",    _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM9_MODE),      REG_PM_PWM9_MODE_MASK},
    {"PM_PWM10_MODE",   _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM10_MODE),     REG_PM_PWM10_MODE_MASK},
    {"PM_UART1_MODE",   _RIUA_16BIT(PMSLEEP_BANK,REG_PM_UART1_MODE),     REG_PM_UART1_MODE_MASK},
    {"PM_VID_MODE_1",   _RIUA_16BIT(PMSLEEP_BANK,REG_PM_VID_MODE),       REG_PM_VID_MODE_MASK},
    {"PM_VID_MODE_2",   _RIUA_16BIT(PMSLEEP_BANK,REG_PM_VID_MODE),       REG_PM_VID_MODE_MASK},
    {"PM_VID_MODE_3",   _RIUA_16BIT(PMSLEEP_BANK,REG_PM_VID_MODE),       REG_PM_VID_MODE_MASK},
    {"PM_SD_CDZ_MODE",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_SD_CDZ_MODE),    REG_PM_SD_CDZ_MODE_MASK},
    {"PM_LED_MODE",     _RIUA_16BIT(PMSLEEP_BANK,REG_PM_LED_MODE),       REG_PM_LED_MODE_MASK},
    {"PM_TTL_MODE_1",   _RIUA_16BIT(CHIPTOP_BANK,REG_TTL_MODE),          REG_TTL_MODE_MASK},
    {"PM_TTL_MODE_2",   _RIUA_16BIT(CHIPTOP_BANK,REG_TTL_MODE),          REG_TTL_MODE_MASK},
    {"PM_IRIN_MODE",    _RIUA_16BIT(PMSLEEP_BANK,REG_PM_IR_IS_GPIO),     REG_PM_IR_IS_GPIO_MASK},
    {"PM_SAR_MODE",     _RIUA_16BIT(SAR_BANK,0x11),                      0x3F},
    {"PM_USB_MODE",     _RIUA_16BIT(UTMI0_BANK,REG_UTMI0_GPIO_EN),       REG_UTMI0_GPIO_EN_MASK},
};

//==============================================================================
//
//                              FUNCTIONS
//
//==============================================================================

//------------------------------------------------------------------------------
//  Function    : _HalCheckPin
//  Description :
//------------------------------------------------------------------------------
static S32 _HalCheckPin(U32 padID)
{
    if (GPIO_NR <= padID) {
        return FALSE;
    }
    return TRUE;
}

static void _HalSARGPIOWriteRegBit(u32 u32RegOffset, bool bEnable, U8 u8BitMsk)
{
    if (bEnable)
        _GPIO_R_BYTE(_RIUA_8BIT(SAR_BANK, u32RegOffset)) |= u8BitMsk;
    else
        _GPIO_R_BYTE(_RIUA_8BIT(SAR_BANK, u32RegOffset)) &= (~u8BitMsk);
}

static void _HalPadDisablePadMux(U32 u32PadModeID)
{
    if (_GPIO_R_WORD_MASK(m_stPadModeInfoTbl[u32PadModeID].u32ModeRIU, m_stPadModeInfoTbl[u32PadModeID].u32ModeMask)) {
        _GPIO_W_WORD_MASK(m_stPadModeInfoTbl[u32PadModeID].u32ModeRIU, 0, m_stPadModeInfoTbl[u32PadModeID].u32ModeMask);
    }
}

static S32 HalPadSetMode_General(U32 u32PadID, U32 u32Mode)
{
    U32 u32RegAddr = 0;
    U16 u16RegVal  = 0;
    U8  u8ModeIsFind = 0;
    U16 i = 0;

    for (i = 0; i < sizeof(m_stPadMuxTbl)/sizeof(struct stPadMux); i++)
    {
        if (u32PadID == m_stPadMuxTbl[i].padID)
        {
            u32RegAddr = _RIUA_16BIT(m_stPadMuxTbl[i].base, m_stPadMuxTbl[i].offset);

            if (u32Mode == m_stPadMuxTbl[i].mode)
            {
                u16RegVal = _GPIO_R_WORD_MASK(u32RegAddr, 0xFFFF);
                u16RegVal &= ~(m_stPadMuxTbl[i].mask);
                u16RegVal |= m_stPadMuxTbl[i].val; // CHECK Multi-Pad Mode

                _GPIO_W_WORD_MASK(u32RegAddr, u16RegVal, 0xFFFF);

                u8ModeIsFind = 1;
#if (ENABLE_CHECK_ALL_PAD_CONFLICT == 0)
                break;
#endif
            }
            else
            {
                u16RegVal = _GPIO_R_WORD_MASK(u32RegAddr, m_stPadMuxTbl[i].mask);

                if (u16RegVal == m_stPadMuxTbl[i].val)
                {
                    printk(KERN_INFO"[Padmux]reset PAD%d(reg 0x%x:%x; mask0x%x) t0 %s (org: %s)\n",
                           u32PadID,
                           m_stPadMuxTbl[i].base,
                           m_stPadMuxTbl[i].offset,
                           m_stPadMuxTbl[i].mask,
                           m_stPadModeInfoTbl[u32Mode].u8PadName,
                           m_stPadModeInfoTbl[m_stPadMuxTbl[i].mode].u8PadName);
                    if (m_stPadMuxTbl[i].val != 0)
                    {
                        _GPIO_W_WORD_MASK(u32RegAddr, 0, m_stPadMuxTbl[i].mask);
                    }
                    else
                    {
                        _GPIO_W_WORD_MASK(u32RegAddr, m_stPadMuxTbl[i].mask, m_stPadMuxTbl[i].mask);
                    }
                }
            }
        }
    }

    return (u8ModeIsFind) ? 0 : -1;
}

static S32 HalPadSetMode_MISC(U32 u32PadID, U32 u32Mode)
{
    switch(u32PadID)
    {
    /* PM_GPIO4 */
    case PAD_PM_GPIO4:
        if (u32Mode == PINMUX_FOR_GPIO_MODE)
        {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_GPIO_PM_LOCK), 0xBABE, REG_PM_GPIO_PM_LOCK_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_LINK_WKINT2GPIO4), 0, REG_PM_LINK_WKINT2GPIO4_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_GPIO_PM4_INV), 0, REG_PM_GPIO_PM4_INV_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_UART1_MODE), 0, REG_PM_UART1_MODE_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(CHIPTOP_BANK,REG_TTL_MODE), 0, REG_TTL_MODE_MASK);
        }
        else if (u32Mode == REG_PM_UART1_MODE)
        {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_GPIO_PM_LOCK), 0xBABE, REG_PM_GPIO_PM_LOCK_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_LINK_WKINT2GPIO4), 0, REG_PM_LINK_WKINT2GPIO4_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_GPIO_PM4_INV), 0, REG_PM_GPIO_PM4_INV_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_UART1_MODE), 0, REG_PM_UART1_MODE_MASK);
        }
        else if (u32Mode == PINMUX_FOR_TTL_MODE_1)
        {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_GPIO_PM_LOCK), 0xBABE, REG_PM_GPIO_PM_LOCK_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_LINK_WKINT2GPIO4), 0, REG_PM_LINK_WKINT2GPIO4_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_GPIO_PM4_INV), 0, REG_PM_GPIO_PM4_INV_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_UART1_MODE), 0, REG_PM_UART1_MODE_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(CHIPTOP_BANK,REG_TTL_MODE), BIT6, REG_TTL_MODE_MASK);
        }
        else if (u32Mode == PINMUX_FOR_TTL_MODE_2)
        {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_GPIO_PM_LOCK), 0xBABE, REG_PM_GPIO_PM_LOCK_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_LINK_WKINT2GPIO4), 0, REG_PM_LINK_WKINT2GPIO4_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_GPIO_PM4_INV), 0, REG_PM_GPIO_PM4_INV_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_UART1_MODE), 0, REG_PM_UART1_MODE_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(CHIPTOP_BANK,REG_TTL_MODE), BIT7, REG_TTL_MODE_MASK);
        }
        else
        {
            return -1;
        }
        break;

    /* SAR */
    case PAD_SAR_GPIO0: /* reg_sar_aisel; reg[1422]#5 ~ #0=0b */
        if (u32Mode == PINMUX_FOR_GPIO_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL_8BIT, 0, REG_SAR_CH0_AISEL);
        }
        else if (u32Mode == PINMUX_FOR_SAR_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL_8BIT, REG_SAR_CH0_AISEL, REG_SAR_CH0_AISEL);
        }
        else
        {
            return -1;
        }
        break;
    case PAD_SAR_GPIO1:
        if (u32Mode == PINMUX_FOR_GPIO_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL_8BIT, 0, REG_SAR_CH1_AISEL);
        }
        else if (u32Mode == PINMUX_FOR_SAR_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL_8BIT, REG_SAR_CH1_AISEL, REG_SAR_CH1_AISEL);
        }
        else
        {
            return -1;
        }
        break;
    case PAD_SAR_GPIO2:
        if (u32Mode == PINMUX_FOR_GPIO_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL_8BIT, 0, REG_SAR_CH2_AISEL);
        }
        else if (u32Mode == PINMUX_FOR_SAR_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL_8BIT, REG_SAR_CH2_AISEL, REG_SAR_CH2_AISEL);
        }
        else
        {
            return -1;
        }
        break;
    case PAD_SAR_GPIO3:
        if (u32Mode == PINMUX_FOR_GPIO_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL_8BIT, 0, REG_SAR_CH3_AISEL);
        }
        else if (u32Mode == PINMUX_FOR_SAR_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL_8BIT, REG_SAR_CH3_AISEL, REG_SAR_CH3_AISEL);
        }
        else
        {
            return -1;
        }
        break;

    /* lan-top */
    case PAD_ETH_RN:
        if (u32Mode == PINMUX_FOR_GPIO_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), BIT0, BIT0);
        }
        else if (u32Mode == PINMUX_FOR_ETH_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), 0, BIT0);
        }
        else
        {
            return -1;
        }
        break;
    case PAD_ETH_RP:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), BIT1, BIT1);
        }
        else if (u32Mode == PINMUX_FOR_ETH_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), 0, BIT1);
        }
        else
        {
            return -1;
        }
        break;
    case PAD_ETH_TN:
        if (u32Mode == PINMUX_FOR_GPIO_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), BIT2, BIT2);
        }
        else if (u32Mode == PINMUX_FOR_ETH_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), 0, BIT2);
        }
        else
        {
            return -1;
        }
        break;
    case PAD_ETH_TP:
        if (u32Mode == PINMUX_FOR_GPIO_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), BIT3, BIT3);
        }
        else if (u32Mode == PINMUX_FOR_ETH_MODE)
        {
            _HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE_2);
            _HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE_2);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), 0, BIT3);
        }
        else
        {
            return -1;
        }
        break;

    /* UTMI */
    case PAD_USB_DM:
    case PAD_USB_DP:
        if (u32Mode == PINMUX_FOR_GPIO_MODE)
        {
            //_HalPadDisablePadMux(PINMUX_FOR_TEST_IN_MODE);
            //_HalPadDisablePadMux(PINMUX_FOR_TEST_OUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_GPIO_EN), REG_UTMI0_GPIO_EN_MASK, REG_UTMI0_GPIO_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_CLK_EXTRA0_EN), ~REG_UTMI0_CLK_EXTRA0_EN_MASK, REG_UTMI0_CLK_EXTRA0_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_REG_PDN), ~REG_UTMI0_REG_PDN_MASK, REG_UTMI0_REG_PDN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_FL_XVR_PDN), ~REG_UTMI0_FL_XVR_PDN_MASK, REG_UTMI0_FL_XVR_PDN_MASK);
        }
        else if (u32Mode == PINMUX_FOR_USB_MODE)
        {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_GPIO_EN), ~REG_UTMI0_GPIO_EN_MASK, REG_UTMI0_GPIO_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_CLK_EXTRA0_EN), REG_UTMI0_CLK_EXTRA0_EN_MASK, REG_UTMI0_CLK_EXTRA0_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_REG_PDN), REG_UTMI0_REG_PDN_MASK, REG_UTMI0_REG_PDN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_FL_XVR_PDN), REG_UTMI0_FL_XVR_PDN_MASK, REG_UTMI0_FL_XVR_PDN_MASK);
        }
        else
        {
            return -1;
        }
        break;
    default:
        break;
    }

    return 0;
}

//------------------------------------------------------------------------------
//  Function    : HalPadSetVal
//  Description :
//------------------------------------------------------------------------------
S32 HalPadSetVal(U32 u32PadID, U32 u32Mode)
{
    if (FALSE == _HalCheckPin(u32PadID)) {
        return FALSE;
    }

    if (u32PadID == PAD_PM_GPIO4 ||
        (u32PadID >= PAD_SAR_GPIO0 && u32PadID <= PAD_USB_DP))
    {
        return HalPadSetMode_MISC(u32PadID, u32Mode);
    }
    else
    {
        return HalPadSetMode_General(u32PadID, u32Mode);
    }
}
//------------------------------------------------------------------------------
//  Function    : HalPadSet
//  Description :
//------------------------------------------------------------------------------
S32 HalPadSetMode(U32 u32Mode)
{
    U32 u32PadID;
    U16 k = 0;

    for (k = 0; k < sizeof(m_stPadMuxTbl)/sizeof(struct stPadMux); k++)
    {
        if (u32Mode == m_stPadMuxTbl[k].mode)
        {
            u32PadID = m_stPadMuxTbl[k].padID;
            if (HalPadSetMode_General( u32PadID, u32Mode) < 0)
            {
                return -1;
            }
        }
    }

    return 0;
}
